/*----------------------------------------------------------------------------
 * Name     : main.c
 * Purpose  : 
 * Note(s)  : 
 *----------------------------------------------------------------------------
 * This file is part of the uVision/ARM development tools.
 * This software may only be used under the terms of a valid, current,
 * end user licence from KEIL for a compatible version of KEIL software
 * development tools. Nothing else gives you the right to use this software.
 *
 * This software is supplied "AS IS" without warranties of any kind.
 *
 * Copyright (c) 2012 Keil - An ARM Company. All rights reserved.
 *----------------------------------------------------------------------------*/

#include    <stdio.h>
#include		<math.h>
#include    <string.h>
#include    "main.h"
#include    "timer.h"
#include    "lcm.h"
#include    "led.h"
#include    "button.h"
#include		"bc3603.h"
#include		"bc3603spi.h"
#include		"bc3603test.h"
#include		"bch.h"
#include		"rf_parameter.h"


// <<< Use Configuration Wizard in Context Menu >>>

//	<h> MCU Sleep Mode Configuration
//		<o> MCU Mode Select
//			<0=> Normal
//			<1=> Sleep
//			<2=> Deep Sleep 1
//			<3=> Deep Sleep 2
#define _MCU_SLEEP_MODE_			(0)
//	</h>

/*----------------------------------------------
							Main Flag Definition
 ----------------------------------------------*/
union
{
	u16	value;
	struct
	{
		u16	button_change: 1;						// button flag
		u16 lcm_updata_falg:1;					// LCM need to updata flag
		u16	rf_tx_ing:1;								// RF TXing flag
		u16	rf_tx_stop:1;								// RF stop flag
		u16	rf_rx_testing:1;						// RF BER test flag
    u16:15;		
	}bits;
}mflag;

/*----------------------------------------------
					RF working parameter Definition
 ----------------------------------------------*/
struct __attribute__((packed))
{
	// RF parameter
	u8	tx_payload_len;
	u8	rx_payload_len;
	u8*	p_txbuf;
	u8*	p_rxbuf;
	
	// Proc parameter
	u8	rf_step;
	u16	tx_success_number;
	u16	tx_fail_number;
	u16	rx_success_number;
	u16	rx_fail_number;
	u8	rx_rcv_rssi;
	u8	rx_env_rssi;
}  RadioParam;


u32   TimePeriod;
u8		key_status;
u8		IRQStatus;
u8		TxPayload[256],RxPayload[256];
u16   TimePeriod_EnvRSSI;
uc8	PN9SyncWord[] = {0x81,0x4A,0xF2,0xEE,0x07,0x3A,0x4F,0x5D};

/*----------------------------------------------------------------------------
  Main Program
 *----------------------------------------------------------------------------*/
int main (void)
{
	/* ----------------------- System initial ------------------------ */
	CKCU_Configuration();
	MCU_gpio_Configuration();
	SystickConfigure();
#if (_RETARGET == 1)
	RETARGET_Configuration();
#endif	
#if (_MCU_SLEEP_MODE_ != 0)
	BTN_WakeupConfig(ENABLE);		
	EXTI_WakeupEventIntConfig(ENABLE);
#endif
	
	/* timer 1ms-flag initial */
	timer_operate.bits.t1ms = FALSE;
	
	/* ------------------------ LCM initial ------------------------- */
	lcm_bus_configure();
	lcm_configure();
	lcm_put_map((u8 *)BestComm_Logo);
	lcm_put8x16_string((sc8 *)"BC3603",0,10,DSP_NORMAL);
	lcm_put8x16_string((sc8 *)" Demo ",1,10,DSP_NORMAL);
	lcm_put8x16_string((sc8 *)"Ver1.0",2,10,DSP_NORMAL);
	lcm_put6x8_string((sc8 *)"22/12/12",6,13,DSP_NORMAL);	
	
	/* --------------- LED/Button IO/Time Base initial --------------- */ 
	BTN_InterfaceConfigure();	
	LED_InterfaceConfigure();
	
	/* ----------------------- BC3603 initial ----------------------- */ 
	/* BC3603 Interface configuration */
	BC3603_InterfaceConfigure();
	/* BC3603 default register configuration */
	if( BC3603_DeInit() != SUCCESS)
	{
		lcm_put8x16_string((sc8 *)"RF DeInit Error",0,0,DSP_NORMAL);
		while(1) delay_ms(100);
	}	
	/* BC3603 RF parameter configuration */
	if( BC3603_Initial() != SUCCESS)
	{
		lcm_put8x16_string((sc8 *)"RF Initial Error",0,0,DSP_NORMAL);
		while(1) delay_ms(100);
	}	
	/* reset variable */
	mflag.bits.rf_tx_ing					 = 0;
	mflag.bits.button_change			 = 0;
	mflag.bits.lcm_updata_falg		 = 0;
	RadioParam.rf_step 						 = 0;
	RadioParam.rx_env_rssi			 	 = 255;
	TimePeriod_EnvRSSI						 = 200;
	
	/* ---------------------- LCM display delay ---------------------- */ 
  /* delay 1s */
	delay_ms(1000);
	lcm_clear();
	LCM_Display_Init();
	
	/* ---------------------------- Loop ---------------------------- */ 
	while(1)
	{
		/* ------------------- Timer PROC ------------------- */ 
		if(timer_operate.bits.t1ms)
		{
			/* common */
			timer_operate.bits.t1ms = FALSE;
			if(TimePeriod) TimePeriod--;			
			if(TimePeriod_EnvRSSI) TimePeriod_EnvRSSI--;	
			/* scan button */
			mflag.bits.button_change = BTN_ScanButtonStatus();
			/* LED PROC */
			LED_FlashProcess();
		}	
		
		/* -------------------- KEY PROC -------------------- */ 
		if(mflag.bits.button_change)
		{
			mflag.bits.button_change = FALSE;
			key_status = BTN_GetButtonStatus();
			switch(key_status)
			{
				case Key1:
				case Key2:
				case Key3:
				case Key4:
					if(mflag.bits.rf_rx_testing)
					{
						/* -------------- STOP RX BER -------------- */ 
						mflag.bits.rf_rx_testing = 0;
						/* RF step resrt */
						RadioParam.rf_step = 1;
					}
					else
					{
						/* -------------- START RX BER -------------- */ 
						mflag.bits.rf_rx_testing = 1;
						/* RF step resrt */
						RadioParam.rf_step = 2;;
					}
					break;
				
				default:
					break;
			}
		}
		
		/* -------------------- RF PROC -------------------- */ 
		/* RF STEP */
		switch(RadioParam.rf_step)
		{
			case 1:
				/* -------------- STOP RX BER TEST -------------- */ 
				/* exit direct-rx mode */
				BC3603_DirectModeInactive();
				/* RF enter deep sleep */
				BC3603_EnterDeepSleep();	
				/* RF step resrt */
				RadioParam.rf_step = 0;
				/* LCM updata */
				LCM_Display_Updata();
				/* LED1 off */
				LED_AlwaysOFF(LED1);
				break;
			
			case 2:
				/* -------------- START RX BER TEST -------------- */
				/* RF enter light sleep */
				BC3603_EnterLightSleep();		
				/* wait crystal ready */
				BC3603_WaitCrystalReady();
				/* set SYNC to be the same as PN9-data */
				BC3603_SetSyncWord((u8 *)PN9SyncWord,SYNCWORD_MAX_SIZE);
				/* RF enter direct-rx mode */
				/* RF GIO2 -> RF_GIO_TRXD */
				/* RF GIO3 -> RF_GIO_RBCLK */		
				BC3603_DirectModeActive(DIRECT_RX);	
				/* RF GIO2 -> RF_GIO_INPUT */
				BC3603_SetGIOFunction(RF_GIO2,RF_GIO_INPUT);
				/* RF GIO3 -> RF_GIO_TRXD */
				BC3603_SetGIOFunction(RF_GIO3,RF_GIO_TRXD);
				/* RF step resrt */
				RadioParam.rf_step = 0;
				/* LCM updata */
				LCM_Display_Updata();
				/* LED1 on */
				LED_AlwaysON(LED1);
				break;
			
			default:
				break;
		}
		
		/* updata environment rssi value */
		if(mflag.bits.rf_rx_testing)
		{
			if(TimePeriod_EnvRSSI == 0 )
			{
				/* set updata interval time */
				TimePeriod_EnvRSSI = 200;
				/* resd value */
				RadioParam.rx_env_rssi += BC3603_GetRSSIValue();
				RadioParam.rx_env_rssi /= 2;
				/* LCM updata */
				mflag.bits.lcm_updata_falg = 1;		
			}		
		}
		else
		{
			/* stop updata*/
			
		}
		
		/* -------------------- LCM PROC -------------------- */ 
		if(mflag.bits.lcm_updata_falg)
		{
			mflag.bits.lcm_updata_falg = 0;
			/* updata LCM display */
			LCM_Display_Updata();
		}

	}
}
/*****************************************************************************************************
* @brief  carrier test item initial.
* @retval None
*******************************************************************************************************/
ErrStatus BC3603_Initial(void)	
{
	u8							Syncw[SYNCWORD_MAX_SIZE];
	u8							SyncBch[SYNCWORD_MAX_SIZE];
	RF_InitTypeDef	RF_InitStruct;
	
	/* Config by <rf_parameter.h> */
	RF_InitStruct.RFBand = (RF_BAND_Enum)_RF_FREQ_BAND_;
	RF_InitStruct.Frequency = _RF_FREQUENCY_;
	RF_InitStruct.TxPower = (RF_TXPWR_Enum)_RF_TX_POWER_;
	RF_InitStruct.DataRate = (RF_DATARATE_Enum)_RF_DATA_RATE_;
	RF_InitStruct.TxPremLength = _TX_PREAMBLE_LENGTH_;
	RF_InitStruct.TxPremClass = (RF_PMPATWD_Enum)_TX_PREAMBLE_CLASS_;
	RF_InitStruct.TxPremPattern = _TX_PREAMBLE_PATTERN_;	
	RF_InitStruct.SyncWordWidth = (RF_SYNCLEN_Enum)_SYNCWORD_WIDTH_;
	RF_InitStruct.TrailerField = (_TRAILER_FIELD_ != 0)?ENABLE:DISABLE;
	RF_InitStruct.HeadField = (RF_PLHF_Enum)_HEAD_WIDTH_;
	RF_InitStruct.LengthField = (_LENGTH_FIELD_ != 0)?ENABLE:DISABLE;
	RF_InitStruct.CRCField = (RF_CRCFMT_Enum)_CRC_FIELD_;
	RF_InitStruct.CRCInv = (RF_CRCPOL_Enum)_CRC_INVERTED_;
	RF_InitStruct.CRCBitOD = (RF_BITOD_Enum)_CRC_BIT_ORDER_;
	RF_InitStruct.CRCByteOD = (RF_BYTEOD_Enum)_CRC_BYTE_ORDER_;
	RF_InitStruct.CRCSeed = _CRC_SEED_;	
	RF_InitStruct.ManchesterCode = (_MANCHESTER_CODE_ != 0)?ENABLE:DISABLE;
	RF_InitStruct.HammingCode = (_HAMMING_CODE_ != 0)?ENABLE:DISABLE;
	RF_InitStruct.HeadAddrExamine = (_EXAMINE_PLHAC_ != 0)?ENABLE:DISABLE;
	RF_InitStruct.WhiteningCodec = (RF_WHTFMT_Enum)_WHITENING_CODEC_;
	RF_InitStruct.WhiteningSeed = _WHITENING_SEED_;
	RF_InitStruct.ReceiverMode = (_RECEIVER_MODE_ != 0)?RECEIVER_CONTINUE:RECEIVER_SINGLE;
	RF_InitStruct.AutomaticACK = (_AUTOMATIC_ACK_ != 0)?ENABLE:DISABLE;
	RF_InitStruct.TimeUnit = (RF_ATRTU_Enum)_ATRK_TIME_UNIT_;
	
	/* RF enter light sleep */
	BC3603_EnterLightSleep();	
	/* wait crystal ready */
	if(BC3603_WaitCrystalReady() != SUCCESS)	return(ERROR);
	/* RF set crysatal tune */
	BC3603_SetCrystalCload(_CYL_COARSE_TUNE_,_CYL_FINE_TUNE_);
	/* RF packet */
	if(BC3603_Configures(&RF_InitStruct) != SUCCESS) return(ERROR);
	BC3603_SetReceiverLength(_RECEIVER_PDLEN_);
	BC3603_SetHeaderAddress(_HEAD_ADDRESS_);	
	
	/* RF SYNC */
	*((u16 *)(&Syncw[0])) = _SYNCWORD_LOW_ & 0xFFFF;
	*((u16 *)(&Syncw[2])) = _SYNCWORD_LOW_ >> 16;
	*((u16 *)(&Syncw[4])) = _SYNCWORD_HIGH_ & 0xFFFF;
	*((u16 *)(&Syncw[6])) = _SYNCWORD_HIGH_ >> 16;
#if	(_SYNCWORD_BCH_ == 1)
	BCH_SyncwordCalculate(RF_InitStruct.SyncWordWidth,Syncw,SyncBch);
	BC3603_SetSyncWord(SyncBch,SYNCWORD_MAX_SIZE);
#else
	BC3603_SetSyncWord(Syncw,SYNCWORD_MAX_SIZE);	
#endif
	/* LIRC calibartion */
	if(BC3603_LIRCCalibration() == ERROR) return(ERROR);
	/* RF enter deep sleep */
	BC3603_EnterDeepSleep();	
	return(SUCCESS);
}
/*****************************************************************************************************
* @brief  carrier test item initial.
* @retval None
*******************************************************************************************************/
ErrStatus BC3603_ParameterUpdate(void)	
{
	return(SUCCESS);
}

/*****************************************************************************************************
* @brief  LCM display initial.
* @retval None
*******************************************************************************************************/
const char LCM_DR_SHOW[][4]=
{
	"  2",
	" 10",
	" 50",
	"125",
	"250",
};
const char LCM_BAND_SHOW[][4]=
{
	"315",
	"433",
	"470",
	"868",
	"915",
};
void LCM_Display_Init(void)	
{
	lcm_put8x8_string((s8 *)"RX_BER_Mode",0,0,DSP_NORMAL);
	lcm_put8x8_string((s8 *)">>BER_Test",1,6,DSP_NORMAL); 
	
	lcm_put8x8_string((s8 *)"000MHz | 000kbps",2,0,DSP_NORMAL); 
	lcm_put8x8_string((s8 *)LCM_BAND_SHOW[_RF_FREQ_BAND_],2,0,DSP_NORMAL); 
	lcm_put8x8_string((s8 *)LCM_DR_SHOW[_RF_DATA_RATE_],2,9,DSP_NORMAL); 
	
	// LCU updata status
	LCM_Display_Updata();
}

/*****************************************************************************************************
* @brief  LCM display updata
* @retval None
*******************************************************************************************************/
void LCM_Display_Updata(void)	
{
	u8 lcm_buffer[22];
	
	// rx receive packet rssi
	sprintf((char *)lcm_buffer,"Rssi:-%03d",RadioParam.rx_env_rssi);
	lcm_put8x8_string((s8 *)lcm_buffer,5,0,DSP_NORMAL);
	
	lcm_put8x8_string((s8 *)"    ",7,12,DSP_CLEAR); 
	if(mflag.bits.rf_rx_testing)
		// RUN status
		lcm_put8x8_string((s8 *)"RUN",7,13,DSP_INVERT);
	else
		lcm_put8x8_string((s8 *)"STOP",7,12,DSP_INVERT); 
}

/*********************************************************************************************************
 * @brief   This function handles WAKE UP interrupt.
 * @retval  None
 ************************************************************************************************************/
#if (_MCU_SLEEP_MODE_ != 0)
void WKUP_IRQHandler(void)
{
	NVIC_ClearPendingIRQ(EVWUP_IRQn);
	NVIC_DisableIRQ(EVWUP_IRQn);
  /*--------------------------------------------------------------------------------------------------------*/
  /* DSB instruction is added in this function to ensure the write operation which is for clearing interrupt*/
  /* flag is actually completed before exiting ISR. It prevents the NVIC from detecting the interrupt again */
  /* since the write register operation may be pended in the internal write buffer of Cortex-Mx when program*/
  /* has exited interrupt routine. This DSB instruction may be masked if this function is called in the     */
  /* beginning of ISR and there are still some instructions before exiting ISR.                             */
  /*--------------------------------------------------------------------------------------------------------*/
  __DSB();
}
#endif
/*****************************************************************************************************
* @brief  Configures the system clocks.
* @retval None
*******************************************************************************************************/
void MCU_gpio_Configuration(void)
{
#if	(MDEBIO_ENABLE == 1)
	AFIO_GPxConfig(MDBIO_P0_AFIO_PORT, MDBIO_P0, MDBIO_P0_AFIO_FUN);
	AFIO_GPxConfig(MDBIO_P1_AFIO_PORT, MDBIO_P1, MDBIO_P1_AFIO_FUN);

	GPIO_DirectionConfig(MDBIO_P0_PORT,MDBIO_P0,GPIO_DIR_OUT);
	MDBIO_P0_CLR();
	GPIO_DirectionConfig(MDBIO_P1_PORT,MDBIO_P1,GPIO_DIR_OUT);
	MDBIO_P1_CLR();
#endif	
}

/*******************************************************************************************************
 * @brief  Configures the system clocks.
 * @retval None
 *******************************************************************************************************/
/*
//  <h> Clock Control Configuration
//  	<h> Communication
//    	<q0.12> EBI
//    	<q1.0> I2C0		<q1.1> I2C1
//    	<q1.25> I2S
//    	<q1.24> SCI0	<q1.27> SCI1
//    	<q1.4> SPI0 	<q1.5> SPI1
//    	<q1.8> USART0	<q1.9> USART1
//    	<q1.10> UART0  <q1.11> UART1
//    	<q0.10>  USB
//  	</h>
//  	<h> IO
//    	<q0.16> GPIO Port A <q0.17>  GPIO Port B <q0.18>  GPIO Port C <q0.19>  GPIO Port D
//    	<q1.14> AFIO
//    	<q1.15> EXTI
//  	</h>
//  	<h> System
//    	<q2.24> ADC
//    	<q0.11> CKREF
//    	<q0.13>  CRC
//    	<q2.22> CMP
//    	<q0.4>  PDMA
//    	<q2.6> PWRCU
//  	</h>
//  	<h> Timer
//    	<q2.16> BFTM0 <q2.17> BFTM1
//    	<q2.28> SCTM0 <q2.29> SCTM1 <q2.30> SCTM2 <q2.31> SCTM3
//    	<q2.8> GPTM0 <q2.9> GPTM1
//    	<q2.0> MCTM0
//    	<q2.6> RTC   <q2.4> WDT
//  	</h>
//  </h>
*/
#define	_CKCU_AHBCCR_		(0x000F2810)
#define	_CKCU_APBCR0_		(0x0000CF33)
#define	_CKCU_APBCR1_		(0xF1430341)

void CKCU_Configuration(void)
{
	CKCU_PeripClockConfig_TypeDef cken;
  
   /* APB Peripheral Clock Selection to CK_AHB or CK_AHB/4 */
   HT_CKCU->APBPCSR0 = 0;           /* Reset value of APBPCSR0   */
   HT_CKCU->APBPCSR1 = 0;           /* Reset value of APBPCSR1   */ 

   /* Configure the CK_ADC prescaler(CK_ADC) = HCLK / 1 */
   CKCU_SetADCPrescaler(CKCU_ADCPRE_DIV1);   
	/* Configure the AHB/APB clock control */
	cken.Reg[0] = _CKCU_AHBCCR_;
	cken.Reg[1] = _CKCU_APBCR0_;
	cken.Reg[2] = _CKCU_APBCR1_;	
   CKCU_PeripClockConfig(cken,ENABLE);   
}

